"""Environment related utils."""

import numpy as np
from mojo.elements import Site


def get_random_sites(
    sites: np.ndarray[Site], amount: int, step: int = 1, segment: int = 0
) -> list[Site]:
    """Get random subset of Sites."""
    sites = sites[::step]
    if segment:
        if segment > 0:
            sites = sites[:segment]
        else:
            sites = sites[segment:]
    return np.random.choice(sites, size=amount, replace=False).tolist()


def get_random_points_on_plane(
    amount: int,
    origin: np.ndarray,
    extents: np.ndarray,
    spacing: float,
    random_offset_bounds: np.ndarray = np.zeros(3),
) -> list[np.ndarray]:
    """Get random set of points on plane within 2D extents."""
    assert len(extents) >= 2
    x_count = int(extents[0] * 2 // spacing)
    y_count = int(extents[1] * 2 // spacing)
    assert x_count * y_count >= amount
    center = np.array([(x_count - 1) * spacing / 2, (y_count - 1) * spacing / 2, 0])
    ref_points = []
    for x in range(x_count):
        for y in range(y_count):
            ref_points.append((x, y))
    ids = np.random.choice(len(ref_points), size=amount, replace=False).tolist()
    ref_points = [ref_points[i] for i in ids]
    points = []
    for point in ref_points:
        random_offset = np.random.uniform(-random_offset_bounds, random_offset_bounds)
        points.append(
            np.array([point[0], point[1], 0]) * spacing
            + origin
            + random_offset
            - center
        )
    return points
